/*
 * StraightMoveEffectInfo.h
 *
 * Created 9/3/2009 By Johnny Huynh
 *
 * Version 00.00.01 9/3/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef STRAIGHT_MOVE_EFFECT_INFO_H
 #define STRAIGHT_MOVE_EFFECT_INFO_H
 
 template <typename T> class StraightMoveEffectInfo;
 
 #include "global.h"
 
 #include "collisionEntry.h"
 #include "MoveEffectInfo.h"
 #include "EffectTask.h"
 #include "StraightMovementInfo.h"
 #include "MovementInfoCollection.h"
 #include "pointerTo.h"
 
 #include "MoveInfo.h"
 
 /**
  * Class specification for StraightMoveEffectInfo
  */
 template <typename T>
 class StraightMoveEffectInfo : public MoveEffectInfo<T>
 {
 // Protected Static Functions
 protected:
    static inline void process_straight_move_back_effect( EffectInfo<T>* effect_info_Ptr, Object<T>* into_object_Ptr, const CollisionEntry& c_entry );
 
 // Data Members
 private:
    T _distance;
    
 // Local Functions
 public:
    StraightMoveEffectInfo( const T distance, const double duration, EffectFunc* effect_func = &StraightMoveEffectInfo<T>::process_straight_move_back_effect );
    StraightMoveEffectInfo( const StraightMoveEffectInfo<T>& straight_move_effect_info );
    virtual ~StraightMoveEffectInfo();
    inline StraightMoveEffectInfo<T>& operator=( const StraightMoveEffectInfo<T>& straight_move_effect_info );
    inline T get_move_distance() const;
    inline void set_move_distance( const T distance );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** STATIC FUNCTIONS **/
 
 /**
  * process_straight_move_back_effect() applies the specified relative straight move back effect (pointed to) onto the 
  * specified object (pointed to).
  * The effect_info_Ptr must be a pointer to a StraightMoveEffectInfo.
  *
  * @param (EffectInfo<T>*) effect_info_Ptr
  * @param (Object<T>*) into_object_Ptr
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void StraightMoveEffectInfo<T>::process_straight_move_back_effect( EffectInfo<T>* effect_info_Ptr, 
                                                                      Object<T>* into_object_Ptr, 
                                                                      const CollisionEntry& c_entry )
 {  
    nassertv( effect_info_Ptr != NULL );
    nassertv( into_object_Ptr != NULL );
    
    StraightMoveEffectInfo<T>* move_effect_info_Ptr( dynamic_cast<StraightMoveEffectInfo<T>*>(effect_info_Ptr) );
    
    // The contact normal is set in the function, handle_collision(), of CollisionHandlerEventNode.h
    VECTOR3_TYPE normal_vector( c_entry.get_contact_normal( into_object_Ptr->get_parent() ) );
    // or try get_contact_normal(), which is the vector to the "from" object's position from the "into" object's position
    //printf("normal vector: %1f, %2f\n", normal_vector.get_x(), normal_vector.get_y());
    //VECTOR2_TYPE direction( normal_vector.get_x(), normal_vector.get_y() );
    //VECTOR2_TYPE direction( -normal_vector.get_x(), -normal_vector.get_y() );
    //Vector::normalize( direction );
    
    // The surface normal if the normal vector of the contact point on the "into" object
    //VECTOR3_TYPE normal_vector( c_entry.get_surface_normal( into_object_Ptr->get_parent() ) );
    VECTOR2_TYPE direction( -normal_vector.get_x(), -normal_vector.get_y() );
    Vector::normalize( direction );
    
    const std::string& from_object_key( c_entry.get_from_node_path().get_tag( "Object Key" ) );
    PT(Object<T>) from_object_Ptr( global::_world_Ptr->get_map_area()->get_object( from_object_key ) );
    // if the into_object is actually a from_object colliding into the Object causing this effect,
    // then reverse the movement direction
    if ( into_object_Ptr->get_key() == from_object_Ptr->get_key() )
        direction = -direction;
    
    //printf("direction: %1f, %2f\n", direction.get_x(), direction.get_y());
    MovementInfoCollection<T> movement_info_collection;
    PT(MovementInfo<T>) movement_info_Ptr( new StraightMovementInfo<T>( 
                                           VECTOR3_TYPE( direction.get_x(), direction.get_y(), ZERO ),
                                           move_effect_info_Ptr->get_move_distance(), 
                                           move_effect_info_Ptr->get_move_duration() ) );
    movement_info_collection.add( movement_info_Ptr );
    //printf("distance: %1f\n", move_effect_info_Ptr->get_distance());
    //printf("duration: %1f\n", move_effect_info_Ptr->get_duration());
    PT(MoveInfo<T>) move_info( new MoveInfo<T>( movement_info_collection ) );
    PT(EffectTask<T>) effect_task( new EffectTask<T>( into_object_Ptr, move_info ) );
    effect_task->add_to_task_manager();
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 StraightMoveEffectInfo<T>::StraightMoveEffectInfo( const T distance, const double duration, EffectFunc* effect_func )
                           : MoveEffectInfo<T>( duration, effect_func ),
                             EffectInfo<T>( effect_func ),
                             _distance( distance )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 StraightMoveEffectInfo<T>::StraightMoveEffectInfo( const StraightMoveEffectInfo<T>& straight_move_effect_info )
                           : MoveEffectInfo<T>( move_effect_info ),
                             EffectInfo<T>( move_effect_info ),
                             _distance( move_effect_info._distance )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 StraightMoveEffectInfo<T>::~StraightMoveEffectInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified StraightMoveEffectInfo to this StraightMoveEffectInfo.
  *
  * @param (const StraightMoveEffectInfo<T>&) move_effect_info
  * @return StraightMoveEffectInfo<T>&
  */
 template <typename T>
 inline StraightMoveEffectInfo<T>& StraightMoveEffectInfo<T>::operator=( const StraightMoveEffectInfo<T>& straight_move_effect_info )
 {
    MoveEffectInfo<T>::operator=( straight_move_effect_info );
    // EffectInfo<T>::operator=( straight_move_effect_info );
    _distance = move_effect_info._distance;
    
    return *this;
 }
 
 /**
  * get_move_distance() returns the straight movement distance.
  *
  * @return T
  */
 template <typename T>
 inline T StraightMoveEffectInfo<T>::get_move_distance() const
 {
    return _distance;
 }
 
 /**
  * set_move_distance() sets the straight movement distance to the specified distance.
  *
  * @param (const T) distance
  */
 template <typename T>
 inline void StraightMoveEffectInfo<T>::set_move_distance( const T distance )
 {
    _distance = distance;
 }
 
 #endif // STRAIGHT_MOVE_EFFECT_INFO_H